/*
    Utility functions for bogodisk
    Copyright (C) 2012-2018 Darrick Wong

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "bogodisk.h"
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sched.h>
#include <stdio.h>
#include <sys/mount.h>
#include <unistd.h>
#include <sys/syscall.h>
#include <sys/time.h>
#include <sys/resource.h>

uint64_t get_number(const char *string)
{
	int sz = strlen(string);
	double result = atof(string);
	char last = string[sz - 1];

	switch (last) {
		case 'k':
		case 'K':
			result *= 1024;
			break;
		case 'm':
		case 'M':
			result *= 1048576;
			break;
		case 'g':
		case 'G':
			result *= 1073741824;
			break;
	}

	return result;
}

void get_size_and_block_size(const char *fname, uint64_t *size,
			     uint32_t *block_size, int *is_blkdev)
{
	uint32_t blksz = 0;
	int fd, is_block = 0;
	uint64_t sz = 0;
#ifdef __USE_LARGEFILE64
	struct stat64 si64;
#else
	struct stat si;
#endif

	/* Try to get sz/blksz info from stat */
#ifdef __USE_LARGEFILE64
	if (!stat64(fname, &si64)) {
		sz = si64.st_size;
		blksz = si64.st_blksize;
		if (S_ISBLK(si64.st_mode))
			is_block = 1;
	}
#else
	if (stat(fname, &si)) {
		sz = si.st_size;
		blksz = si64.st_blksize;
		if (S_ISBLK(si.st_mode))
			is_block = 1;
	}
#endif

	/* Block device?  Query it directly. */
	if (is_block) {
		fd = open(fname, O_RDONLY);
		if (fd < 0) {
			perror(fname);
			goto close_file;
		}
#ifdef BLKGETSIZE64
		if (ioctl(fd, BLKGETSIZE64, &sz))
			perror(fname);
#endif
#ifdef BLKSSZGET
		if (ioctl(fd, BLKSSZGET, &blksz))
			perror(fname);
#endif
	
close_file:
		close(fd);
	}

	if (blksz)
		goto found_blksz;

	/* Still no block size?  Assume page size. */
#ifdef _SC_PAGESIZE
	blksz = sysconf(_SC_PAGESIZE);
	if (blksz > 0)
		goto found_blksz;
#endif
#ifdef PAGESIZE
	blksz = sysconf(PAGESIZE);
	if (blksz > 0)
		goto found_blksz;
#endif
	blksz = 4096;

found_blksz:
	if (is_blkdev)
		*is_blkdev = is_block;
	if (size)
		*size = sz;
	if (block_size)
		*block_size = blksz;
	return;
}

int seed_random(void) {
	int fp;
	long seed;

	fp = open("/dev/urandom", O_RDONLY);
	if (fp < 0) {
		perror("/dev/urandom");
		return 0;
	}

	if (read(fp, &seed, sizeof(seed)) != sizeof(seed)) {
		perror("read random seed");
		close(fp);
		return 0;
	}

	close(fp);
	srand(seed);

	return 1;
}

uint64_t get_randnum_align(uint64_t min, uint64_t max, uint64_t align) {
	return (min + (uint64_t)((double)(max - min) * (rand() / (RAND_MAX + 1.0)))) &
		~(align - 1);
}

/* Slurp files from linux/ioprio.h */
#ifdef SYS_ioprio_set
 #ifndef IOPRIO_PRIO_VALUE
enum {
	IOPRIO_CLASS_NONE,
	IOPRIO_CLASS_RT,
	IOPRIO_CLASS_BE,
	IOPRIO_CLASS_IDLE,
};

enum {
        IOPRIO_WHO_PROCESS = 1,
        IOPRIO_WHO_PGRP,
        IOPRIO_WHO_USER,
};

#define IOPRIO_CLASS_SHIFT		(13)
#define IOPRIO_PRIO_VALUE(class, data)	(((class) << IOPRIO_CLASS_SHIFT) | data)

long ioprio_set(int which, int who, int ioprio)
{
	return syscall(SYS_ioprio_set, which, who, ioprio);
}
 #endif
#endif

int bump_priority(void)
{
	int res;

	res = ioprio_set(IOPRIO_WHO_PROCESS, getpid(),
			 IOPRIO_PRIO_VALUE(IOPRIO_CLASS_RT, 7));
	if (res) {
		perror("ioprio_set");
		return -1;
	}

	res = setpriority(PRIO_PROCESS, 0, -20);
	if (res) {
		perror("setpriority");
		return -1;
	}

	return 0;
}
