import java.io.*;
import javax.swing.tree.*;
import javax.swing.*;

public class ClassReader implements ElementInfo, Runnable {
	//variables
	private String fname = "";
	private int magic[] = new int[2], version[] = new int[2];
	private int constant_pool_count, access_flags, this_class, super_class;
	private InterfaceInfo interfaces;
	private FieldInfo fields;
	private MethodInfo methods;
	private AttributeInfo attributes;
	private ConstantPoolInfo constants;

	private JTree tree;
	private StatusKeeper stats, stats2;

	//constants
	public static final int ACC_PUBLIC = 0x0001;
	public static final int ACC_PRIVATE = 0x0002;
	public static final int ACC_PROTECTED = 0x0004;
	public static final int ACC_STATIC = 0x0008;
	public static final int ACC_FINAL = 0x0010;
	public static final int ACC_SYNCHRONIZED = 0x0020;
	public static final int ACC_THREADSAFE = 0x0040;
	public static final int ACC_TRANSIENT = 0x0080;
	public static final int ACC_NATIVE = 0x0100;
	public static final int ACC_INTERFACE = 0x0200;
	public static final int ACC_ABSTRACT = 0x0400;

	private Thread th;

	public ClassReader(String s, JTree t, StatusKeeper x, StatusKeeper x2){
		fname = s;
		tree = t;
		stats = x;
		stats2 = x2;
		th = new Thread(this);
		th.setPriority(Thread.MIN_PRIORITY+2);
		th.start();
	}

	public void run() {
		stats.initProgressBar();
		stats.showProgressBar();
		stats.setMax(16);

		ReadClass();

		stats.hideProgressBar();
		stats.hideWindow();
	}

	private void ReadClass() throws ClassFormatError {
		/*
		NOTES: Reading the Class File
		Establish a DataInputStream
		Check the file to make sure it's OK.
		Read attributes!  More on this later...
		*/
		try {

		//As of 8/6, there are 16 parts to loading a file.
		//Set the progress bar's max to 16,
		//and increment as necessary.
		stats.setStatus("Opening class \""+fname+"\"...");
			FileInputStream inFile = new FileInputStream(fname);
			DataInputStream is = new DataInputStream(new BufferedInputStream(inFile));

		stats.setProgressText("Verifying class file integrity...", 1);

			magic[0] = is.readUnsignedShort();
			magic[1] = is.readUnsignedShort();

			if(magic[0] != 0xCAFE || magic[1] != 0xBABE) throw new ClassFormatError("The file you selected is not a valid Java class file.");

		stats.setProgressText("Reading class version...", 2);

			version[0] = is.readUnsignedShort();
			version[1] = is.readUnsignedShort();

		stats.setProgressText("Reading constant pool...", 3);
			constants = new ConstantPoolInfo(is, stats2);

		stats.setProgressText("Reading class info...", 4);
			access_flags = is.readUnsignedShort();
			this_class = is.readUnsignedShort();
			super_class = is.readUnsignedShort();

		stats.setProgressText("Reading Interface info...", 5);
			interfaces = new InterfaceInfo(constants, is, stats2);

		stats.setProgressText("Reading Field pool...", 6);
			fields = new FieldInfo(constants, is, stats2);

		stats.setProgressText("Reading & Disassembling Methods...", 7);
			methods = new MethodInfo(constants, is, stats2);

		stats.setProgressText("Reading class attributes...", 8);
			attributes = new AttributeInfo(constants, is);

			//Now that we've loaded ALL of the data, let's dump it all!
		stats.setProgressText("Formatting class info...", 9);
			DefaultMutableTreeNode tnRoot;
			tnRoot = new DefaultMutableTreeNode("Class: "+fname);
			tnRoot.add(Describe());

		stats.setProgressText("Formatting constant pool...", 10);
			tnRoot.add(constants.Describe());

		stats.setProgressText("Formatting Interface info...", 11);
			tnRoot.add(interfaces.Describe());

		stats.setProgressText("Formatting Field pool...", 12);
			tnRoot.add(fields.Describe());

		stats.setProgressText("Formatting Method disassembly...", 13);
			tnRoot.add(methods.Describe());

		stats.setProgressText("Formatting class attributes...", 14);
			tnRoot.add(attributes.Describe());

		stats.setProgressText("Changing tree contents...", 15);
			//Make a new, empty model and assign it to the three.
			DefaultTreeModel treeModel = new DefaultTreeModel(tnRoot);
			tree.setModel(treeModel);
		stats.setProgressText("Class disassembly complete.", 16);

		} catch (IOException e) {
			e.printStackTrace();
		}

	}
	public DefaultMutableTreeNode Describe(){
		DefaultMutableTreeNode tnGenInfo = new DefaultMutableTreeNode("General Class Info");
		//add magic number
		tnGenInfo.add(
			new DefaultMutableTreeNode("Magic Number: 0x"+
				Integer.toString(magic[0], 16)+
				Integer.toString(magic[1], 16)
			)
		);
		//add class version
		tnGenInfo.add(
			new DefaultMutableTreeNode("Class Version: "+
				Integer.toString(version[1], 10)+"."+
				Integer.toString(version[0], 10)
			)
		);
		tnGenInfo.add(
			new DefaultMutableTreeNode("Access Specifiers: "+
				listAccessFlags(access_flags)
			)
		);
		tnGenInfo.add(
			new DefaultMutableTreeNode("this: "+constants.describe(this_class))
		);
		tnGenInfo.add(
			new DefaultMutableTreeNode("parent: "+constants.describe(super_class))
		);
		return tnGenInfo;
	}
	public static String listAccessFlags(int f){
		StringBuffer buf = new StringBuffer("");

		if ( (f & ACC_PUBLIC)	   != 0 ) buf.append("PUBLIC ");
		if ( (f & ACC_PRIVATE)	  != 0 ) buf.append("PRIVATE ");
		if ( (f & ACC_PROTECTED)	!= 0 ) buf.append("PROTECTED ");
		if ( (f & ACC_STATIC)	   != 0 ) buf.append("STATIC ");
		if ( (f & ACC_FINAL)		!= 0 ) buf.append("FINAL ");
		if ( (f & ACC_SYNCHRONIZED) != 0 ) buf.append("SYNCHRONIZED ");
		if ( (f & ACC_THREADSAFE)   != 0 ) buf.append("THREADSAFE ");
		if ( (f & ACC_TRANSIENT)	!= 0 ) buf.append("TRANSIENT ");
		if ( (f & ACC_NATIVE)	   != 0 ) buf.append("NATIVE ");
		if ( (f & ACC_INTERFACE)	!= 0 ) buf.append("INTERFACE ");
		if ( (f & ACC_ABSTRACT)	 != 0 ) buf.append("ABSTRACT ");
		if ( buf.length() < 5			) buf.append("DEFAULT" );
		 return buf.toString();
	}
}
