/*
 * time.c -- Figure out how much time it has been since the last update.
 * Copyright (C) 2006 Darrick Wong
 */
#include <stdio.h>
#include <time.h>
#include "ugh.h"

static struct timespec times[2];
static struct timespec *old_time, *new_time;
static int fudged = 0;
static clockid_t clock_type = CLOCK_MONOTONIC;

/* Initialize time data gathering */
int init_time(void)
{
	struct timespec junk;

	old_time = times;
	new_time = times + 1;

	if (clock_gettime(clock_type, &junk))
		clock_type = CLOCK_REALTIME;
	clock_gettime(clock_type, &junk);
	*old_time = junk;
	*new_time = junk;

	return 1;
}

/* Translate nanoseconds to timespec. */
struct timespec ns_to_timespec(uint64_t x)
{
	struct timespec t;
	
	t.tv_sec = x / 1000000000;
	t.tv_nsec = x % 1000000000;

	return t;
}

/* Translate timespec to nanoseconds. */
uint64_t timespec_to_ns(struct timespec *t)
{
	return (uint64_t)t->tv_nsec + (uint64_t)t->tv_sec * 1000000000;
}

/* Record call time */
void update_time(void)
{
	struct timespec *tmp;
	uint64_t x;
	int needs_fudge;
	
	tmp = new_time;
	new_time = old_time;
	old_time = tmp;

	clock_gettime(clock_type, new_time);
	needs_fudge = !elapsed_time();

	if (fudged && !needs_fudge) {
		x = timespec_to_ns(new_time);
		x -= fudged * 1000000;
		*new_time = ns_to_timespec(x);
		fudged = 0;
	}

	if (needs_fudge) {
		fudged++;
		x = timespec_to_ns(new_time);
		x += 1000000;
		*new_time = ns_to_timespec(x);
	}
}

/* Calculate time in ms between two timespecs */
static uint64_t calc_time_delta(struct timespec *new,
	struct timespec *old)
{
	uint64_t n, o;

	n = (uint64_t)new->tv_sec * 1000 +
		(uint64_t)new->tv_nsec / 1000000;
	o = (uint64_t)old->tv_sec * 1000 +
		(uint64_t)old->tv_nsec / 1000000;

	return n - o;
}

/* Return elapsed time */
uint64_t elapsed_time(void)
{
	return calc_time_delta(new_time, old_time);
}
